-- ============================================================================ 
-- === TrailerWeight.Lua 
-- === MOD my [LSMT] Modding Team 
-- === LS25 / FS25
-- === Script by [LSMT] BaTt3RiE @ 2025 
-- === Ver 0.1.2.1 (Beta) - Multiplayer-fähig angepasst
-- ============================================================================

TrailerWeight = {}

DEBUG = false

local function debugPrint(...)
    if DEBUG then
        print(...)
    end
end

function TrailerWeight.prerequisitesPresent(specializations)
    return true
end

function TrailerWeight.registerFunctions(vehicle)
    SpecializationUtil.registerFunction(vehicle, "onWeighingTriggerCallback", TrailerWeight.onWeighingTriggerCallback)
    SpecializationUtil.registerFunction(vehicle, "updateWeightDisplay", TrailerWeight.updateWeightDisplay)
    SpecializationUtil.registerFunction(vehicle, "setWeightDisplay", TrailerWeight.setWeightDisplay)
end

function TrailerWeight.registerEventListeners(vehicle)
    SpecializationUtil.registerEventListener(vehicle, "onLoad", TrailerWeight)
    SpecializationUtil.registerEventListener(vehicle, "onDelete", TrailerWeight)
    SpecializationUtil.registerEventListener(vehicle, "onDraw", TrailerWeight)
    SpecializationUtil.registerEventListener(vehicle, "onUpdate", TrailerWeight)
end

function TrailerWeight.onLoad(self)
    self.spec_TrailerWeight = {}
    local spec = self.spec_TrailerWeight
    spec.currentWeight = 0
    spec.triggerVehicleNodes = {}
    spec.weightOffset = 14000
    spec.liftAxleThreshold = 5000
    spec.dollyweight = 4200
    spec.triggerNodePath ="0>0|1|11"

    spec.hudX = 0.5
    spec.hudY = 0.9
    spec.fontSize = 0.03
    spec.textAlignment = RenderText.ALIGN_CENTER
    spec.timeSinceLastUpdate = 0

    spec.triggers = {}
    local triggerNodePath = spec.triggerNodePath
    local trigger1 = I3DUtil.indexToObject(self.components, triggerNodePath)
    if trigger1 ~= nil then
        table.insert(spec.triggers, trigger1)
        addTrigger(trigger1, "onWeighingTriggerCallback", self)
    else
        debugPrint("Kein Trigger-Node gefunden, Trailerwiegefunktion wird ohne Trigger fortgeführt.")
    end
end

function TrailerWeight.onDelete(self)
    local spec = self.spec_TrailerWeight
    if spec.triggers then
        for _, trigger in ipairs(spec.triggers) do
            removeTrigger(trigger)
        end
        spec.triggers = nil
    end
end

function TrailerWeight.onWeighingTriggerCallback(self, triggerId, otherId, onEnter, onLeave, onStay, otherShapeId)
    local spec = self.spec_TrailerWeight
    if onEnter then
        if spec.triggerVehicleNodes[otherId] then
            spec.triggerVehicleNodes[otherId] = spec.triggerVehicleNodes[otherId] + 1
        else
            spec.triggerVehicleNodes[otherId] = 1
        end
    elseif onLeave then
        if spec.triggerVehicleNodes[otherId] then
            spec.triggerVehicleNodes[otherId] = spec.triggerVehicleNodes[otherId] - 1
            if spec.triggerVehicleNodes[otherId] <= 0 then
                spec.triggerVehicleNodes[otherId] = nil
            end
        end
    end
    -- Nur auf dem Server ausführen, da hier die Berechnung erfolgt
    if g_server then
        self:updateWeightDisplay()
    end
end

-- Gewicht wird ausschließlich auf dem Server berechnet
function TrailerWeight.onUpdate(self, dt)
    if g_server then
        local spec = self.spec_TrailerWeight
        spec.timeSinceLastUpdate = spec.timeSinceLastUpdate + dt
        if spec.timeSinceLastUpdate >= 500 then  -- 500 ms = 10 Sekunde
            self:updateWeightDisplay()
            spec.timeSinceLastUpdate = 0
        end
    end
end

function TrailerWeight.updateWeightDisplay(self)
    local spec = self.spec_TrailerWeight
    local totalWeight = 0
    for nodeId, _ in pairs(spec.triggerVehicleNodes) do
        if entityExists(nodeId) then
            local vehicle = g_currentMission:getNodeObject(nodeId)
            if vehicle ~= nil and vehicle.getTotalMass then
                totalWeight = totalWeight + vehicle:getTotalMass(true)
            end
        else
            spec.triggerVehicleNodes[nodeId] = nil
        end
    end

    local computedWeight = totalWeight * 1000 -- Umrechnung von Tonnen in kg
    local adjustedWeight = computedWeight - spec.weightOffset

    -- Extra Gewicht hinzufügen, falls das direkt angehängte Fahrzeug keinen eigenen Motor hat
    local attacherVehicle = self:getAttacherVehicle()
    if attacherVehicle ~= nil and attacherVehicle.getIsMotorStarted == nil then
        adjustedWeight = adjustedWeight + spec.dollyweight
    end

    if adjustedWeight < 0 then
        adjustedWeight = 0
    end
        self:setWeightDisplay(adjustedWeight)
end

function TrailerWeight.setWeightDisplay(self, weight)
    if self.spec_TrailerWeight.currentWeight ~= weight then
        self.spec_TrailerWeight.currentWeight = weight
        -- Gewichtsänderung nur auf dem Server synchronisieren
        TrailerWeightEvent.sendEvent(self, weight)

    end
end

function TrailerWeight.onDraw(self)
    if DEBUG then
        local spec = self.spec_TrailerWeight
        local weightText = string.format("%d Offset", spec.currentWeight)
        renderText(spec.hudX, spec.hudY, spec.fontSize, weightText, 1, 1, 1, 1, spec.textAlignment)
    end
end

--------------------------------------------------------------------------------
-- Neuer Netzwerk-Event für TrailerWeight zur Synchronisation des Gewichts
--------------------------------------------------------------------------------
TrailerWeightEvent = {}
TrailerWeightEvent_mt = Class(TrailerWeightEvent, Event)

function TrailerWeightEvent.emptyNew()
    local self = Event.new(TrailerWeightEvent_mt)
    return self
end

function TrailerWeightEvent.new(vehicle, weight)
    local self = TrailerWeightEvent.emptyNew()
    self.vehicle = vehicle
    self.weight = weight
    return self
end

function TrailerWeightEvent:readStream(streamId, connection)
    self.vehicle = NetworkUtil.readNodeObject(streamId)
    self.weight = streamReadFloat32(streamId)
    self:run(connection)
end

function TrailerWeightEvent:writeStream(streamId, connection)
    NetworkUtil.writeNodeObject(streamId, self.vehicle)
    streamWriteFloat32(streamId, self.weight)
end

function TrailerWeightEvent:run(connection)
    if self.vehicle then
        self.vehicle.spec_TrailerWeight.currentWeight = self.weight
        if not connection:getIsServer() then
            g_server:broadcastEvent(self, false)
        end
    end
end

function TrailerWeightEvent.sendEvent(vehicle, weight)
    if g_server then
        g_server:broadcastEvent(TrailerWeightEvent.new(vehicle, weight), nil, nil, vehicle)
    else
        g_client:getServerConnection():sendEvent(TrailerWeightEvent.new(vehicle, weight))
    end
end

InitEventClass(TrailerWeightEvent, "TrailerWeightEvent")
